; -----------------------------------------------------------------------------
;  GBSIM.ASM
; -----------------------------------------------------------------------------
;  Dieses Modul simuliert den Prozessor des Gameboys. Wenn es einmal ntig ist,
;  kann es auf Z80 erweitert werden.
; -----------------------------------------------------------------------------
;  Die aktuelle Version interpretiert dazu den Maschienencode. Jeder Befehl wird
;  als Index in eine Opcode-Tabelle interpretiert. In der Opcode-Tabelle
;  ist der Index der richtigen Emulationsroutine gespeichert. Jede Routine
;  "wei" welche Parameter der Opcode braucht und inkrementiert den PC
;  entsprechend.
; -----------------------------------------------------------------------------
IDEAL
P386    ; Das vereinfacht die Sache ein bischen...
; -----------------------------------------------------------------------------
;  Alle GB-Register haben ein Equivalent bei den PC-Registern.
; -----------------------------------------------------------------------------
;  DS/ES -> Ram
;  SP = DI  PC = SI   AF = AX
;  HL = BX  BC = CX   DE = DX

reg_sp EQU DI
reg_pc EQU SI
reg_af EQU AX
reg_hl EQU BX
reg_bc EQU CX
reg_de EQU DX

ind_sp EQU EDI
ind_pc EQU ESI
ind_af EQU EAX
ind_hl EQU EBX
ind_bc EQU ECX
ind_de EQU EDX

reg_a EQU AL
reg_f EQU AH
reg_h EQU BH
reg_l EQU BL
reg_b EQU CH
reg_c EQU CL
reg_d EQU DH
reg_e EQU DL

STRUC TRegStruc         ;  Hier werden die Register gesichert.
rAF     DW 0
rBC     DW 0
rDE     DW 0
rHL     DW 0
rSP     DW 0
rPC     DW 0
ENDS

;  Das User-Interface:
GLOBAL aAsmData:NEAR    ;  Variablen der CPU-Emulation
GLOBAL CallGB:NEAR      ;  GameBoy Callgate
GLOBAL WriteSeg:WORD
GLOBAL ReadSeg:WORD
GLOBAL Joypad:BYTE

SEGMENT Code PARA PUBLIC 'CODE'
ASSUME CS:Code,DS:Code

LABEL aAsmData NEAR
RegStruc        TRegStruc <>    ;  Register
Flags           DW 0            ;  Ereignisse whrend der Emualtion
IE              DB 0            ;  Interrupt-Flag
WriteSeg        DW 0            ;  64 K for Writing
ReadSeg         DW 0            ;  64 K for Reading
MMUPage         DB 0            ;  Eingeblendete MMU-Page
JoyPad          DB 0            ;  Aktuelle Joypad-Position
IOReg           DW 0            ;  IO-Adresse
PScrInfo        DD 0            ;  Information about the Screen

CycleCount      DD 0

;  Ereignisse:
opTimerInt      EQU 1   ;  Timer-Int wurde ausgefhrt
opSerInt        EQU 2   ;
opKeyInt        EQU 4   ;
opError         EQU 8   ;  Fehler (Illegal Opcode)
opHalt          EQU 16  ;  HLT-Befehl
opSingleStep    EQU 32  ;  Einzelschritt-Modus
opMMUAccess     EQU 64  ;  Schreibzugriff auf $2000

TInt            EQU 8
KBInt           EQU 9

; -----------------------------------------------------------------------------
;  Sichern und laden der Flag-Register
; -----------------------------------------------------------------------------
;  Ich kann es nicht glauben ! Vorher wurde dies durch eine komplizierte
;  Pushf/Popf/Push/Pop-kombination gemacht. Durch Zufall fand ich aber diese
;  beiden OpCodes. Es ist eben doch gut, da die 80x86 mit dem Z80 verwandt ist.
; -----------------------------------------------------------------------------

MACRO SaveFlags
		lahf
ENDM

MACRO LoadFlags
		sahf
ENDM

; -----------------------------------------------------------------------------
;  Gameboy-Versionen von Push+Pop
; -----------------------------------------------------------------------------

MACRO GB_Push Value  ;  Oft gebraucht...
		Sub reg_SP,2
		Mov [es:reg_SP],Value
ENDM

MACRO GB_Pop Value
		Mov Value,[es:reg_SP]
		Add reg_SP,2
ENDM

; -----------------------------------------------------------------------------
;  Sichern und Laden der Register ber den Record RegStruc
; -----------------------------------------------------------------------------

MACRO LoadRegs
		Movzx ind_AF,[CS:RegStruc.rAF]
		Movzx ind_BC,[CS:RegStruc.rBC]
		Movzx ind_DE,[CS:RegStruc.rDE]
		Movzx ind_HL,[CS:RegStruc.rHL]
		Movzx ind_SP,[CS:RegStruc.rSP]
		Movzx ind_PC,[CS:RegStruc.rPC]
ENDM

MACRO SaveRegs 
		Mov [CS:RegStruc.rAF],reg_AF
		Mov [CS:RegStruc.rBC],reg_BC
		Mov [CS:RegStruc.rDE],reg_DE
		Mov [CS:RegStruc.rHL],reg_HL
		Mov [CS:RegStruc.rSP],reg_SP
		Mov [CS:RegStruc.rPC],reg_PC
ENDM

; -----------------------------------------------------------------------------
;  Diese Prozeduren werden in den Timer, den Breakpoint und den
;  Tastaturinterrupt eingeschleift um ein Flag zu setzen.
; -----------------------------------------------------------------------------

PROC NewIntKeyb
		Or [CS:Flags],opKeyInt
		DB 0EAh
OldIntKeyb      DD (?)
ENDP

PROC NewIntTimer
		Or [CS:Flags],opTimerInt
		DB 0EAh
OldIntTimer     DD (?)
ENDP

; -----------------------------------------------------------------------------
; Calls a GB-IRQ
; -----------------------------------------------------------------------------

PROC CallIRQ
		Cmp [CS:IE],0               ; Check, if IRQ are enabled
		Je SHORT @@Done
		Test [BYTE PTR DS:0FF0Fh],1
		Jne SHORT @@VLB
		Test [BYTE PTR DS:0FF0Fh],2
		Jne SHORT @@FLI
		Test [BYTE PTR DS:0FF0Fh],4
		Jne SHORT @@TIMER
		Test [BYTE PTR DS:0FF0Fh],8
		Jne SHORT @@SER
		Test [BYTE PTR DS:0FF0Fh],16
		Jne SHORT @@JOYPAD
@@Done:         Ret
@@VLB:          Mov [CS:IE],0                   ; Disable IRQ
		And [BYTE PTR DS:0FF0Fh],NOT 1  ; Clear Flag
		GB_Push reg_PC                  ; Store old adress
		Mov reg_PC,040h                 ; New adress
		Ret
@@FLI:          Mov [CS:IE],0                   ; Disable IRQ
		And [BYTE PTR DS:0FF0Fh],NOT 2  ; Clear Flag
		GB_Push reg_PC                  ; Store old adress
		Mov reg_PC,048h                 ; New adress
		Ret
@@TIMER:        Mov [CS:IE],0                   ; Disable IRQ
		And [BYTE PTR DS:0FF0Fh],NOT 4  ; Clear Flag
		GB_Push reg_PC                  ; Store old adress
		Mov reg_PC,050h                 ; New adress
		Ret
@@SER:          Mov [CS:IE],0                   ; Disable IRQ
		And [BYTE PTR DS:0FF0Fh],NOT 8  ; Clear Flag
		GB_Push reg_PC                  ; Store old adress
		Mov reg_PC,058h                 ; New adress
		Ret
@@JOYPAD:       Mov [CS:IE],0                   ; Disable IRQ
		And [BYTE PTR DS:0FF0Fh],NOT 16 ; Clear Flag
		GB_Push reg_PC                  ; Store old adress
		Mov reg_PC,060h                 ; New adress
		Ret
ENDP


; -----------------------------------------------------------------------------
;  Die Emulation wird ber Callgb gestartet.
;  Diese Funktion ist ein Call-Gate, alle Register werden geladen und die
;  Interrupt-Pointer werden verndert.
;  Jeder Opcode wird dann durch das Macro NextOpcode beendet.
; -----------------------------------------------------------------------------

MACRO NextOpcode
		Movzx BP,[BYTE PTR reg_PC]
		Inc reg_PC
		Shl BP,1
		jmp [WORD PTR CS:OpCodeTable00+BP]
ENDM

PROC CallGB NEAR
		Push DS
		Push ES
		Pushad
		Cli
		Xor AX,AX
		Mov DS,AX
;  Alte Ints sichern:
;               Mov EAX,[Tint*4]
;               Mov [CS:OldIntTimer],EAX
		Mov EAX,[KBint*4]
		Mov [CS:OldIntKeyb],EAX
;  Neue Ints setzen:
		Mov AX,Code                     ;  High-Word von EAX ist Segment
		Shl EAX,16
;               Mov AX,OFFSET NewIntTimer
;               Mov [Tint*4],EAX
		Mov AX,OFFSET NewIntKeyb
		Mov [KBint*4],EAX
		Sti
;  Moduswechsel:
		Mov DS,[CS:ReadSeg]     ;  DS fr alle Lesezugriffe
		Mov ES,[CS:WriteSeg]    ;  ES fr alle Schreibzugriffe

		LoadRegs

		Mov [CS:Cyclecount],0FFh
		Call CallIRQ

;  Ersten Opcode laden und ausfhren:
		NextOpcode
ENDP

; -----------------------------------------------------------------------------
;  Hier wird der Grundzustand des Systems hergestellt um nach TP zurckzukehren.
; -----------------------------------------------------------------------------

PROC Return
		SaveRegs
;               Call Holds_ASM
		Cli
		Xor AX,AX
		Mov DS,AX
;  Alte Ints wiederherstellen
;               Mov EAX,[CS:OldIntTimer]
;               Mov [Tint*4],EAX
		Mov EAX,[CS:OldIntKeyb]
		Mov [KBint*4],EAX
		Xor EAX,EAX
		Sti
		Popad
		Pop ES
		Pop DS
		Ret
ENDP

;  Dieses Makro definiert ein Label mit der Offsetnummer
MACRO OpDef Num
		PROC OpCode&Num NEAR
ENDM

;  Dieses Makro definiert das Ende eines Opcodes und ruft automatisch den
;  nchsten Opcode auf.
MACRO OpEnd Num
LOCAL Leave,Line
		Cmp [CS:Flags],0        ; Irgendwas besonderes ?
		Jnz SHORT Leave
		Dec [DWORD PTR CS:CycleCount]
		Jz SHORT Line
		NextOpCode
Line:           Or [CS:Flags],opTimerInt
Leave:          Jmp NEAR Return         ; Ende !
		ENDP
ENDM

; =============================================================================
;  IO-Verwaltung:
; =============================================================================

Randseed        DD 7654321

PROC Random
		Push EAX
		Mov EAX,089A5h
		IMul EAX,[CS:Randseed]
		Inc EAX
		Mov [DS:0ff04h],AL
		Mov [CS:Randseed],EAX
		Pop EAX
		Ret
ENDP

PROC VStatus
		Push AX
		Mov AL,[BYTE PTR DS:0FF41h]
		Mov AH,AL
		And AH,0FCh
		Inc AL
		And AL,3
		Or AL,AH
		Mov [BYTE PTR DS:0FF41h],AL
		Pop AX
		Ret
ENDP

PROC IO_Read
		Test [BYTE PTR CS:IOReg],080h   ; Test -> IO-Bereich ?
		Jnz SHORT @@No_IO
		Cmp [BYTE PTR CS:IOReg],004h    ; Random-Register ?
		Je SHORT Random
		Cmp [BYTE PTR CS:IOReg],041h    ; Display-Status ?
		Je SHORT VStatus
@@No_IO:        Ret                             ; Register not used
ENDP

PROC Check_Joypad
		push ax
		mov ah,[CS:Joypad]
		test [BYTE PTR DS:0ff00h],010h
		jz SHORT @@No_Low
		mov al,ah
		and al,00fh
		Or [BYTE PTR DS:0ff00h],al
@@No_Low:       test [BYTE PTR DS:0ff00h],020h
		jz SHORT @@No_High
		shr ah,4
		Or [BYTE PTR DS:0ff00h],ah
@@No_High:      Or [BYTE PTR DS:0ff00h],0f0h
		pop ax
		Ret
ENDP

PROC DMA_Transfer
		Push SI
		Push DI
		Push ECX
		Cld
		Movzx SI,[BYTE PTR DS:0ff46h]   ; Source-Adresse
		Shl SI,8
		Mov DI,0FE00h                   ; Dest-Adresse
		Mov ECX,40                      ; 40 DWORDS (=40 Sprites)
		REP Movsd
		Pop ECX
		Pop DI
		Pop SI
		Ret
ENDP

PROC IO_Write
		Test [BYTE PTR CS:IOReg],080h   ; Test -> IO-Bereich ?
		Jnz SHORT @@No_IO
		Cmp [BYTE PTR CS:IOReg],000h    ; Joypad-Register ?
		Je Check_Joypad
		Cmp [BYTE PTR CS:IOReg],046h    ; DMA-Register ?
		Je DMA_Transfer
@@No_IO:        Ret                             ; Register not used
ENDP

MACRO WRITE_STROBE
	Call IO_Write
ENDM

MACRO READ_STROBE
	Call IO_Read
ENDM

MACRO WRITE_CHECK
LOCAL NO_IO,RAM,NO_MMU
	Or BP,BP
	js SHORT RAM
	push BP
	and BP,03000h
	cmp BP,02000h
	jne SHORT NO_MMU
	pop BP
	push BP
	push AX
	mov AL,[ES:BP]
	mov [CS:MMUPage],AL
	or [CS:Flags],opMMUAccess
	pop AX
NO_MMU: pop BP
	Jmp SHORT NO_IO
RAM:    Cmp BP,0ff00h
	js NO_IO
	Mov [cs:IOreg],BP
	Call IO_Write
NO_IO:
ENDM

MACRO READ_CHECK Reg
LOCAL NO_IO
	Cmp reg,0ff00h
	js NO_IO
	Mov [cs:IOReg],reg
	Call IO_Read
NO_IO:
ENDM

; -----------------------------------------------------------------------------
;  Hier werden die Opcodes definiert. Sie werden dann automatisch in die Opcode-
;  tabelle ingefgt.
; -----------------------------------------------------------------------------

;  OpCode NOP
OpDef %000h
OpEnd

;  OpCode HALT
opDef %076h
	Or [CS:Flags],opHalt
OpEnd

; = Load - Group ==============================================================

;  OpCode ld SP,HL
OpDef %0F9h
	Mov reg_SP,reg_HL
OpEnd

;  OpCode LD r,r
Par1=0
IRP Reg1,<reg_b,reg_c,reg_d,reg_e,reg_h,Reg_l,xxx,reg_a>
	Par2=0
	IRP Reg2,<reg_b,reg_c,reg_d,reg_e,reg_h,Reg_l,xxx,reg_a>
		IFDIFI <Reg1>,<Reg2>
			OpDef %040h+Par1*8+Par2
				IFIDNI <Reg1>,<xxx>
					Mov [es:reg_HL],Reg2
					Mov BP,reg_HL
					WRITE_CHECK
				ELSE
					IFIDNI <Reg2>,<xxx>
						READ_CHECK reg_HL
						Mov Reg1,[ds:reg_HL]
					ELSE
						Mov Reg1,Reg2
					ENDIF
				ENDIF
			OpEnd
		ELSEIFDIFI <Reg1>,<xxx>
			OpDef %040h+Par1*8+Par2
			OpEnd
		ENDIF   
		Par2=Par2+1
	ENDM
	Par1=Par1+1
ENDM

;  OpCode LD r,n
Par1=0
IRP Reg1,<reg_b,reg_c,reg_d,reg_e,reg_h,Reg_l,[es:reg_HL],reg_a>
	IFDIFI <Reg1>,<[es:reg_HL]>
		OpDef %006h+par1*8
			Mov Reg1,[reg_PC]
			Inc reg_PC
		OpEnd
	ENDIF
	Par1=Par1+1
ENDM

;  OpCode LD dd,nn
Par1=0
IRP Reg1,<reg_BC,reg_DE,reg_HL,reg_SP>
	OpDef %001h+par1*16
	Mov Reg1,[ds:reg_PC]
	Add reg_PC,2
	OpEnd
	Par1=Par1+1
ENDM

;  OpCode Ld (BC),a
OpDef %002h
	mov BP,reg_BC
	Mov [es:BP],reg_a
	WRITE_CHECK
OpEnd

;  OpCode Ld (DE),a
OpDef %012h
	mov BP,reg_DE
	Mov [es:BP],reg_a
	WRITE_CHECK
OpEnd

;  OpCode Ld a,(BC)
OpDef %00Ah
	READ_CHECK reg_BC
	Mov reg_a,[ds:ind_BC]
OpEnd

;  OpCode Ld a,(DE)
OpDef %01Ah
	READ_CHECK reg_DE
	Mov reg_a,[ds:ind_DE]
OpEnd

;  OpCode Ld (HL),n
OpDef %036h
	Mov BP,AX
	Mov al,[ds:reg_PC]
	Mov [es:reg_HL],al
	Mov AX,BP
	mov BP,reg_HL
	WRITE_CHECK
	Inc reg_PC
OpEnd

; = Jump ======================================================================

;  OpCode Jp (HL)
OpDef %0E9h
	Mov reg_PC,reg_HL
OpEnd

;  OpCode JP nn
OpDef %0C3h
	Mov reg_PC,[ds:reg_PC]
OpEnd

;  OpCode Jp cc,nn
Par1=0
IRP cc,<z,nz,c,nc>
	OpDef %0C2h+Par1*8
		LoadFlags
		J&cc @@Exit
		Mov reg_PC,[ds:reg_PC]
		Jmp SHORT @@Next
@@Exit:         Add reg_PC,2
@@Next: OpEnd
	Par1=Par1+1
ENDM

;  OpCode Jr e
OpDef %018h
	Movsx BP,[BYTE PTR ds:Reg_PC]
	Add reg_PC,BP
	Inc Reg_PC
OpEnd

;  OpCode Jr cc,e
Par1=0
IRP cc,<z,nz,c,nc>
	OpDef %020h+Par1*8
		LoadFlags
		J&cc @@Exit
		Movsx BP,[BYTE PTR ds:Reg_PC]
		Add reg_PC,BP
@@Exit:         Inc reg_PC
	OpEnd
	Par1=Par1+1
ENDM

; = Call Group ================================================================

;  OpCode Call
OpDef %0CDh
	Add reg_PC,2                    ;  Rcksprungadresse berechnen
	GB_Push reg_PC
	Sub reg_PC,2
	Mov reg_PC,[ds:reg_PC]
OpEnd

;  OpCode Call cc,nn
Par1=0
IRP cc,<z,nz,c,nc>
	OpDef %0C4h+Par1*8
		LoadFlags
		J&cc @@Exit
		Add reg_PC,2                    ;  Rcksprungadresse berechnen
		GB_Push reg_PC
		Sub reg_PC,2
		Mov reg_PC,[ds:reg_PC]
		Jmp SHORT @@Next
@@Exit:         Add reg_PC,2
@@Next: OpEnd
	Par1=Par1+1
ENDM

; = Ret =======================================================================

;  OpCode Ret
OpDef %0C9h
	GB_Pop Reg_PC
OpEnd

;  OpCode Ret cc
Par1=0
IRP cc,<z,nz,c,nc>
	OpDef %0C0h+Par1*8
		LoadFlags
		J&cc @@Exit
		GB_Pop reg_PC
@@Exit: OpEnd
	Par1=Par1+1
ENDM

; = Push+Pop Group ============================================================

;  OpCode Pop reg 16
Par1=0
IRP reg,<reg_BC,reg_DE,reg_HL,reg_AF>
	OpDef %0C1h+Par1*16
		GB_Pop reg
	OpEnd
	Par1=Par1+1
ENDM

;  OpCode Push reg 16
Par1=0
IRP reg,<reg_BC,reg_DE,reg_HL,reg_AF>
	OpDef %0C5h+Par1*16
		GB_Push reg
	OpEnd
	Par1=Par1+1
ENDM

; = Restart Group =============================================================

Par1=0
REPT 8
	OpDef %0C7h+Par1*8
		GB_Push Reg_PC
		Mov reg_PC,Par1*8               ;  Neuer PC:=Par1*8
	OpEnd
	Par1=Par1+1
ENDM

; = Gruppe Numerische Befehle =================================================

Par1=0
IRP Operator,<add,adc,sub,sbb,and,xor,or,cmp>
	Par2=0
	IRP Reg,<reg_b,reg_c,reg_d,reg_e,reg_h,Reg_l,[ds:reg_HL],reg_a>
		OpDef %080h+Par1*8+Par2
			LoadFlags
			Operator reg_a,reg
			SaveFlags
		OpEnd
		Par2=Par2+1
	ENDM
	OpDef %0C6h+Par1*8
		LoadFlags
		Operator reg_a,[ds:reg_PC]
		SaveFlags
		Inc reg_PC
	OpEnd
	Par1=Par1+1
ENDM

; = Gruppe Inc / Dec ==========================================================

Par1=0
IRP Operator,inc,dec
	Par2=0
	IRP Reg,<reg_b,reg_c,reg_d,reg_e,reg_h,Reg_l,[BYTE PTR es:reg_HL],reg_a>
		OpDef %004h+Par2*8+Par1
			LoadFlags
			Operator reg
			SaveFlags
		OpEnd
		Par2=Par2+1
	ENDM
	Par2=0
	IRP Reg,<reg_BC,reg_DE,reg_HL,reg_SP>
		OpDef %003h+Par1*8+Par2*16
			Operator reg
		OpEnd
		Par2=Par2+1
	ENDM
	Par1=Par1+1
ENDM

; = Interrupt Flag ============================================================

;  OpCode DI
OpDef %0F3h
	Mov [CS:IE],0
OpEnd

;  OpCode EI
OpDef %0FBh
	Mov [CS:IE],1
	Call CallIRQ
OpEnd

; = OpCode Add HL,ss ==========================================================

Par1=0
IRP Reg,<reg_BC,reg_DE,reg_HL,reg_SP>
	OpDef %009h+par1*16
	LoadFlags
	And reg_f,0feh
	Add reg_HL,reg
	jnc @@no_c
	or reg_f,001h
	SaveFlags
@@no_c: OpEnd
	Par1=Par1+1
ENDM

; = Shiftoperationen ==========================================================

;  OpCode rlca
OpDef %007h
	LoadFlags
	Rol reg_a,1
	SaveFlags
OpEnd

;  OpCode rrca
OpDef %00Fh
	LoadFlags
	Ror reg_a,1
	SaveFlags
OpEnd

;  OpCode rla
OpDef %017h
	LoadFlags
	Rcl reg_a,1
	SaveFlags
OpEnd

;  OpCode rra
OpDef %01Fh
	LoadFlags
	Rcr reg_a,1
	SaveFlags
OpEnd

; = A-Operationen =============================================================

;  OpCode daa
OpDef %027h
	LoadFlags
	daa
	SaveFlags
OpEnd

;  OpCode cpl
OpDef %02Fh
	LoadFlags
	Not reg_a
	SaveFlags
OpEnd

; = Carry-Flag ================================================================

;  OpCode scf
OpDef %037h
	or reg_f,1
OpEnd

;  OpCode ccf
OpDef %03Fh
	xor reg_f,1
OpEnd

; =============================================================================
;  Gameboy Opcodes
; =============================================================================

;  ?? 10 xx  DJNZ offset    STOP  Meaning unknown ?

;  OpCode Ld (nn),a
OpDef %0EAh
	Mov BP,[ds:reg_PC]
	Mov [es:BP],reg_a
	WRITE_CHECK
	Add reg_PC,2
OpEnd

;  OpCode Ld a,(nn)
OpDef %0FAh
	Mov BP,[ds:reg_PC]
	READ_CHECK BP
	Mov reg_a,[DS:BP]
	Add reg_PC,2
OpEnd

;  OpCode add sp,n
OpDef %0E8h
	Movsx BP,[BYTE PTR ds:reg_PC]
	Add reg_SP,BP
	Inc reg_PC
OpEnd

;  OpCode ld (nn),sp
OpDef %008h
	mov bp,[word ptr DS:reg_PC]
	mov [word ptr ES:bp],reg_sp
	Add reg_PC,2
OpEnd

;  lh hl,sp+n
OpDef %0F8h
	Mov reg_HL,reg_SP
	Movsx BP,[BYTE PTR ds:reg_PC]
	Add reg_HL,BP
	Inc reg_PC
OpEnd

;  OpCode reti
OpDef %0D9h
	Mov [CS:IE],1
	GB_Pop reg_PC
OpEnd

;  OpCode stosb_pos (HL),a
OpDef %022h
	Mov [es:reg_HL],reg_a
	Inc reg_HL
OpEnd

;  OpCode lodsb_pos a,(HL)
OpDef %02Ah
	Mov reg_a,[ds:reg_HL]
	Inc reg_HL
OpEnd

;  OpCode stosb_neg (HL),a
OpDef %032h
	Mov [es:reg_HL],reg_a
	Dec reg_HL
OpEnd

;  OpCode lodsb_neg a,(HL)
OpDef %03Ah
	Mov reg_a,[ds:reg_HL]
	Dec reg_HL
OpEnd

;  OpCode ld (0ff00h+n),a
OpDef %0E0h
	Movzx BP,[BYTE PTR ds:reg_PC]
	Mov [CS:IOReg],BP
	Inc reg_PC
	Mov [ES:0FF00h+BP],reg_a
	WRITE_STROBE
OpEnd

;  OpCode ld a,(0ff00h+n)
OpDef %0F0h
	Movzx BP,[BYTE PTR ds:reg_PC]
	Mov [CS:IOReg],BP
	Inc reg_PC
	READ_STROBE
	Mov reg_a,[DS:0FF00h+BP]
OpEnd

;  OpCode ld (0ff00h+c),a
OpDef %0E2h
	Movzx BP,reg_c
	Mov [ES:0FF00h+BP],reg_a
	Mov [CS:IOReg],BP
	WRITE_STROBE
OpEnd

;  OpCode ld a,(0ff00h+c)
OpDef %0F2h
	Movzx BP,reg_c
	Mov [CS:IOReg],BP
	READ_STROBE
	Mov reg_a,[DS:0FF00h+BP]
OpEnd

; Wird von einer GB-Demo benutzt.
OpDef %0EDh
	Inc reg_pc
OpEnd

; -----------------------------------------------------------------------------
;  Diese Routine "vermittelt" an die CB-Tabelle weiter. Sie ist als Opcode
;  CB definiert.
; -----------------------------------------------------------------------------

OpDef %0CBh
	Movzx BP,[BYTE PTR reg_PC]
	Inc reg_PC
	Shl BP,1
	jmp [WORD PTR CS:OpCodeTableCB+BP]
ENDP

; -----------------------------------------------------------------------------
;  Hier kommen die Opcodes mit dem Prefix CB
; -----------------------------------------------------------------------------

MACRO OpCBDef Num
;       ALIGN 4
	PROC OpCodeCB&Num NEAR
ENDM

; = Shiftoperationen ==========================================================

Par1=0
IRP Operator,<rol,ror,rcl,rcr,shl,sar,xxx,shr>
	Par2=0
	IRP Reg,<reg_b,reg_c,reg_d,reg_e,reg_h,Reg_l,[BYTE PTR es:reg_HL],reg_a>
	IFIDN <Operator>,<xxx>
		OpCBDef %000h+Par1*8+Par2
		ror Reg,4
		OpEnd
	ELSE
		OpCBDef %000h+Par1*8+Par2
		LoadFlags
		Operator Reg,1
		SaveFlags
		OpEnd
	ENDIF
		Par2=Par2+1
	ENDM
	Par1=Par1+1
ENDM

; = Bitoperationen ============================================================

;  Opcode Bit b,r
Par1=0
REPT 8
	Par2=0
	IRP Reg,<reg_b,reg_c,reg_d,reg_e,reg_h,Reg_l,[BYTE PTR ds:reg_HL],reg_a>
		OpCBDef %040h+Par1*8+Par2
		LoadFlags
		Test Reg,(1 SHL Par1)
		SaveFlags
		OpEnd
		Par2=Par2+1
	ENDM
	Par1=Par1+1
ENDM

;  Opcode Set b,r
Par1=0
REPT 8
	Par2=0
	IRP Reg,<reg_b,reg_c,reg_d,reg_e,reg_h,Reg_l,[BYTE PTR es:reg_HL],reg_a>
		OpCBDef %0C0h+Par1*8+Par2
		Or Reg,(1 SHL Par1)
		OpEnd
		Par2=Par2+1
	ENDM
	Par1=Par1+1
ENDM

;  Opcode Res b,r
Par1=0
REPT 8
	Par2=0
	IRP Reg,<reg_b,reg_c,reg_d,reg_e,reg_h,Reg_l,[BYTE PTR es:reg_HL],reg_a>
		OpCBDef %080h+Par1*8+Par2
		And Reg,NOT (1 SHL Par1)
		OpEnd
		Par2=Par2+1
	ENDM
	Par1=Par1+1
ENDM

; -----------------------------------------------------------------------------
;  Dieser Code wird fr alle undefinierten Opcodes aufgerufen.
; -----------------------------------------------------------------------------

PROC Error NEAR
		Or [CS:Flags],opError
		Jmp NEAR Return
ENDP

; -----------------------------------------------------------------------------
;  Dies ist eine Tabelle der Adressen aller Opcode-Emulationsroutinen.
; -----------------------------------------------------------------------------

ALIGN 4

LABEL OpCodeTable00 WORD
MACRO MakeOpCode Num
	IFDEF OpCode&Num
	DW OFFSET OpCode&Num
	ELSE
	DW OFFSET Error
	ENDIF
ENDM
OpNum=0
REPT 256
	MakeOpCode %OpNum
	OpNum=OpNum+1
ENDM

; -----------------------------------------------------------------------------
;  Diese Tabelle enthlt die Befehle mit dem Prefix 0CBh.
; -----------------------------------------------------------------------------

ALIGN 4

LABEL OpCodeTableCB WORD
MACRO MakeOpCodeCB Num
	IFDEF OpCodeCB&Num
	DW OFFSET OpCodeCB&Num
	ELSE
	DW OFFSET Error
	ENDIF
ENDM
OpNum=0
REPT 256
	MakeOpCodeCB %OpNum
	OpNum=OpNum+1
ENDM

; -----------------------------------------------------------------------------
;  Ende ...
; -----------------------------------------------------------------------------

ENDS

END
